import { Router } from 'express';
import { contributorsController } from './controller';
import { authenticateUser, requireVendor } from '@/middlewares/auth';
import { checkFeature } from '@/middlewares/featureGate';
import { vendorService } from '@/modules/vendors/service';

const router = Router();

// Attach vendor to req for downstream use
async function attachVendor(req: any, res: any, next: any) {
  if (req.user) {
    const { data: vendor } = await vendorService.getVendorByUserId(req.user.id);
    req.vendor = vendor || null;
  }
  next();
}

// Contributors endpoints
router.post('/', authenticateUser, requireVendor, attachVendor, checkFeature('contributor_roles'), contributorsController.addContributor);
router.get('/', authenticateUser, requireVendor, attachVendor, checkFeature('contributor_roles'), contributorsController.listContributors);
router.patch('/:id', authenticateUser, requireVendor, attachVendor, checkFeature('contributor_roles'), contributorsController.updateContributor);
router.delete('/:id', authenticateUser, requireVendor, attachVendor, checkFeature('contributor_roles'), contributorsController.deleteContributor);

// Discounts endpoints
router.post('/discounts', authenticateUser, requireVendor, attachVendor, checkFeature('discount_rules'), contributorsController.createDiscount);
router.get('/discounts', authenticateUser, requireVendor, attachVendor, checkFeature('discount_rules'), contributorsController.listDiscounts);
router.get('/discounts/apply', authenticateUser, requireVendor, attachVendor, checkFeature('discount_rules'), contributorsController.applyDiscounts);
router.patch('/discounts/:id', authenticateUser, requireVendor, attachVendor, checkFeature('discount_rules'), contributorsController.updateDiscount);
router.delete('/discounts/:id', authenticateUser, requireVendor, attachVendor, checkFeature('discount_rules'), contributorsController.deleteDiscount);

export default router; 